<?php

namespace App\Http\Controllers\Admin;

use App\Exports\Contest_Question_Export;
use App\Http\Controllers\Controller;
use App\Imports\Contest_Question_Import;
use App\Models\Contest;
use App\Models\Common;
use App\Models\Contest_Leaderboard;
use App\Models\Contest_Question;
use App\Models\Contest_Transaction;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use Exception;

class ContestController extends Controller
{
    private $folder = "contest";
    public $common;
    public function __construct()
    {
        $this->common = new Common;
    }

    public function index(Request $request)
    {
        try {
            $params['data'] = [];

            if ($request->ajax()) {

                $input_search = $request['input_search'];

                $query = Contest::query();

                $input_search = $request['input_search'];
                if ($input_search != null) {
                    $query->where('name', 'LIKE', "%{$input_search}%");
                }
                $data = $query->latest()->get();

                for ($i = 0; $i < count($data); $i++) {

                    $data[$i]['image'] = $this->common->getImage($this->folder, $data[$i]['image']);
                    $data[$i]['total_participants_user'] = Contest_Transaction::where('contest_id', $data[$i]['id'])->count();
                }

                return DataTables()::of($data)
                    ->addIndexColumn()
                    ->addColumn('action', function ($row) {

                        $contest_delete = __('label.delete_contest');

                        $delete = '<form onsubmit="return confirm(\'' . $contest_delete . '\');" method="POST" action="' . route('admin.contests.destroy', [$row->id]) . '">
                        <input type="hidden" name="_token" value="' . csrf_token() . '">
                        <input type="hidden" name="_method" value="DELETE">
                        <button type="submit" class="edit-delete-btn" style="outline: none;"><i class="fa-solid fa-trash-can fa-xl"></i></button></form>';

                        $btn = '<div class="d-flex justify-content-around">';
                        $btn .= '<a href="' . route('admin.contests.show', [$row->id]) . '" class="edit-delete-btn mr-2">';
                        $btn .= '<i class="fa-solid fa-circle-info fa-xl"></i>';
                        $btn .= '</a>';
                        $btn .= '<a href="' . route('admin.contests.question.index', [$row->id]) . '" class="edit-delete-btn mr-2">';
                        $btn .= '<i class="fa-solid fa-circle-question fa-xl"></i>';
                        $btn .= '</a>';
                        $btn .= '<a class="edit-delete-btn mr-2 edit_contests" data-toggle="modal" href="#EditModel" data-id="' . $row->id . '" data-name="' . $row->name . '" data-image="' . $row->image . '" data-total_questions="' . $row->total_questions . '" data-duration="' . $row->duration . '">';
                        $btn .= '<i class="fa-solid fa-pen-to-square fa-xl"></i>';
                        $btn .= '</a>';
                        $btn .= $delete;
                        $btn .= '</a></div>';
                        return $btn;
                    })
                    ->addColumn('status', function ($row) {
                        if ($row['end_date'] >= date("Y-m-d")) {
                            $Label = __('label.not_ended');
                            return "<button type='button' class='info-btn'>$Label</button>";
                        } else {
                            $check = Contest_Leaderboard::where('contest_id', $row['id'])->where('rank', '!=', 0)->first();
                            if ($check) {
                                $showLabel = __('label.leaderboard');
                                $btn = "<a href='" . route('admin.contests.leaderboard', [$row->id]) . "' class='show-btn p-2'>$showLabel</a>";
                            } else {
                                $hideLabel = __('label.make_winner');
                                $btn = "<a href='" . route('admin.contests.makewinner', [$row->id]) . "' class='hide-btn p-2'>$hideLabel</a>";
                            }
                            return $btn;
                        }
                    })
                    ->rawColumns(['action', 'status'])
                    ->make(true);
            }
            return view('admin.contest.index', $params);
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function create()
    {
        try {

            $params['data'] = [];
            return view('admin.contest.add', $params);
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function store(Request $request)
    {
        try {

            $validator = Validator::make($request->all(), [
                'name' => 'required|min:2',
                'start_date' => 'required|date|after_or_equal:today',
                'end_date'   => 'required|date|after_or_equal:start_date',
                'image' => 'required|image|mimes:jpeg,png,jpg|max:5120',
                'entry_fee' => 'required|numeric|min:0',
                'max_participants' => 'required|numeric|min:1',
                'total_prize_users' => 'required|numeric|min:0',
                'total_ranks' => 'required|numeric|max:10|min:1',
                'total_prize_coins' => 'required|numeric|min:0',
                'total_questions' => 'required|numeric|min:1',
                'duration' => 'required|numeric|min:1',
            ]);
            if ($validator->fails()) {
                $errs = $validator->errors()->all();
                return response()->json(['status' => 400, 'errors' => $errs]);
            }

            $contest = new Contest();
            $contest['name'] = $request['name'];
            $file = $request['image'];
            $contest['image'] = $this->common->saveImage($file, $this->folder, 'contest_');
            $contest['start_date'] = $request['start_date'];
            $contest['end_date'] = $request['end_date'];
            $contest['total_questions'] = $request['total_questions'];
            $contest['duration'] = $request['duration'];
            $contest['entry_fee'] = $request['entry_fee'];
            $contest['max_participants'] = $request['max_participants'];
            $contest['total_prize_users'] = $request['total_prize_users'];
            $contest['total_ranks'] = $request['total_ranks'];
            $contest['total_prize_coins'] = $request['total_prize_coins'];
            $contest['prize_json'] = $this->generatePrizeJson($request['total_ranks'], $request['total_prize_users'], $request['total_prize_coins']);
            $contest['status'] = 1;
            if ($contest->save()) {
                return response()->json(['status' => 200, 'success' => __('label.success_add_contest')]);
            } else {
                return response()->json(['status' => 400, 'errors' => __('label.error_add_contest')]);
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    // Price Json
    function generatePrizeJson($maxRankGroups = 10, $totalWinners, $totalPrizeCoins)
    {
        $baseWeights = [];
        $totalWeight = 0;

        // Step 1: Generate weights (1 / rank)
        for ($i = 1; $i <= $totalWinners; $i++) {
            $weight = 1 / $i;
            $baseWeights[] = $weight;
            $totalWeight += $weight;
        }

        // Step 2: Allocate coins based on weights
        $coinPerRank = [];
        for ($i = 0; $i < $totalWinners; $i++) {
            $coins = floor(($baseWeights[$i] / $totalWeight) * $totalPrizeCoins);
            $coinPerRank[] = $coins;
        }

        // Step 3: Fix rounding error (add leftover coins only if needed)
        $diff = $totalPrizeCoins - array_sum($coinPerRank);
        $i = 0;
        while ($diff > 0) {
            $coinPerRank[$i]++;
            $i = ($i + 1) % count($coinPerRank);
            $diff--;
        }

        // Step 4: Group prize distribution
        $grouped = [];
        $i = 0;

        // Always keep rank 1 to 3 individual
        while ($i < min(3, $totalWinners)) {
            $rank = $i + 1;
            $amount = $coinPerRank[$i];
            $grouped[] = [
                'rank' => (string) $rank,
                'winning_coin' => $amount,
                'user_count' => 1,
                'total_coin' => $amount
            ];
            $i++;
        }

        // Group remaining ranks
        while ($i < $totalWinners) {
            $start = $i + 1;
            $remaining = $totalWinners - $i;
            $remainingGroups = max(1, $maxRankGroups - count($grouped));
            $groupSize = max(1, ceil($remaining / $remainingGroups));
            $end = min($start + $groupSize - 1, $totalWinners);

            $userCount = $end - $start + 1;
            $amounts = array_slice($coinPerRank, $i, $userCount);
            $amountPerUser = floor(array_sum($amounts) / $userCount);
            $adjustedTotal = $amountPerUser * $userCount;

            $grouped[] = [
                'rank' => $start === $end ? "$start" : "$start-$end",
                'winning_coin' => $amountPerUser,
                'user_count' => $userCount,
                'total_coin' => $adjustedTotal
            ];

            $i = $end;
        }

        // Step 5: Fix leftover coins ONLY if total distributed is less
        $actualUsedCoins = array_sum(array_column($grouped, 'total_coin'));
        $leftover = $totalPrizeCoins - $actualUsedCoins;

        if ($leftover > 0) {
            $j = 0;
            while ($leftover > 0 && $j < count($grouped)) {
                $grouped[$j]['winning_coin'] += 1;
                $grouped[$j]['total_coin'] = $grouped[$j]['winning_coin'] * $grouped[$j]['user_count'];
                $leftover--;
                $j++;
            }
        }
        return json_encode($grouped);
    }
    public function update(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|min:2',
                'image' => 'image|mimes:jpeg,png,jpg|max:5120',
                'total_questions' => 'required|numeric|min:1',
                'duration' => 'required|numeric|min:1',
            ]);
            if ($validator->fails()) {
                $errs = $validator->errors()->all();
                return response()->json(['status' => 400, 'errors' => $errs]);
            }

            $requestData = $request->all();
            if (isset($requestData['image'])) {
                $file = $requestData['image'];
                $requestData['image'] = $this->common->saveImage($file, $this->folder, 'contest_');

                $this->common->deleteImageToFolder($this->folder, basename($requestData['old_image']));
            }
            unset($requestData['old_image']);

            $data = Contest::updateOrCreate(['id' => $requestData['id']], $requestData);
            if (isset($data['id'])) {
                return response()->json(['status' => 200, 'success' => __('label.success_edit_contest')]);
            } else {
                return response()->json(['status' => 400, 'errors' => __('label.error_edit_contest')]);
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function destroy($id)
    {
        try {

            $data = Contest::where('id', $id)->first();
            if ($data) {

                $this->common->deleteImageToFolder($this->folder, $data['image']);
                $data->delete();
            }
            return redirect()->route('admin.contests.index')->with('success', __('label.contests_delete'));
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function show($id)
    {
        try {

            $params['data'] = Contest::where('id', $id)->first();
            if (isset($params['data']['id'])) {

                $params['data']['image'] = $this->common->getImage($this->folder, $params['data']['image']);
                $params['data']['json'] = json_decode($params['data']['prize_json'], true);

                return view('admin.contest.show', $params);
            } else {
                return redirect()->back()->with('error', __('label.data_not_found'));
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    // Question
    public function question_index($id, Request $request)
    {
        try {

            $params['contest_id'] = $id;
            if ($request->ajax()) {

                $input_search = $request['input_search'];

                $query = Contest_Question::where('contest_id', $id);
                if ($input_search) {
                    $query->where(function ($q) use ($input_search) {
                        $q->where('question', 'LIKE', "%{$input_search}%")
                            ->orWhere('option_a', 'LIKE', "%{$input_search}%")
                            ->orWhere('option_b', 'LIKE', "%{$input_search}%")
                            ->orWhere('option_c', 'LIKE', "%{$input_search}%")
                            ->orWhere('option_d', 'LIKE', "%{$input_search}%");
                    });
                }
                $data = $query->latest()->get();

                $this->common->imageNameToUrl($data, 'image', $this->folder);

                return DataTables()::of($data)
                    ->addIndexColumn()
                    ->addColumn('action', function ($row) {

                        $question_delete = __('label.delete_question');

                        $delete = '<form onsubmit="return confirm(\'' . $question_delete . '\');" method="POST" action="' . route('admin.contests.question.delete', [$row->id]) . '">
                        <input type="hidden" name="_token" value="' . csrf_token() . '">
                        <button type="submit" class="edit-delete-btn" style="outline: none;"><i class="fa-solid fa-trash-can fa-xl"></i></button></form>';

                        $btn = '<div class="d-flex justify-content-around">';
                        $btn .= '<a class="edit-delete-btn mr-2 edit_contests_question" data-toggle="modal" href="#EditModel" data-id="' . $row->id . '" data-question="' . $row->question . '" data-image="' . $row->image . '" data-option_a="' . $row->option_a . '" data-option_b="' . $row->option_b . '" data-option_c="' . $row->option_c . '" data-option_d="' . $row->option_d . '" data-correct_answer="' . $row->correct_answer . '" data-note="' . $row->note . '">';
                        $btn .= '<i class="fa-solid fa-pen-to-square fa-xl"></i>';
                        $btn .= '</a>';
                        $btn .= $delete;
                        $btn .= '</div>';
                        return $btn;
                    })
                    ->addColumn('status', function ($row) {
                        if ($row->status == 1) {
                            $showLabel = __('label.show');
                            return "<button type='button' id='$row->id' onclick='change_status($row->id)' class='show-btn'>$showLabel</button>";
                        } else {
                            $hideLabel = __('label.hide');
                            return "<button type='button' id='$row->id' onclick='change_status($row->id)' class='hide-btn'>$hideLabel</button>";
                        }
                    })
                    ->rawColumns(['action', 'status'])
                    ->make(true);
            }
            return view('admin.contest.question', $params);
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function question_save(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'contest_id' => 'required',
                'question' => 'required',
                'option_a' => 'required',
                'option_b' => 'required',
                'option_c' => 'required',
                'option_d' => 'required',
                'correct_answer' => 'required',
                'image' => 'image|mimes:jpeg,png,jpg|max:5120',
            ]);
            if ($validator->fails()) {
                $errs = $validator->errors()->all();
                return response()->json(['status' => 400, 'errors' => $errs]);
            }

            $question = new Contest_Question();
            $question['contest_id'] = $request['contest_id'];
            $question['image'] = '';
            if (isset($request['image'])) {
                $file = $request['image'];
                $question['image'] = $this->common->saveImage($file, $this->folder, 'contest_question_');
            }
            $question['question'] = $request['question'];
            $question['option_a'] = $request['option_a'];
            $question['option_b'] = $request['option_b'];
            $question['option_c'] = $request['option_c'];
            $question['option_d'] = $request['option_d'];
            $question['correct_answer'] = $request['correct_answer'];
            $question['note'] = $request['note'] ?? '';
            $question['status'] = 1;
            if ($question->save()) {
                return response()->json(['status' => 200, 'success' => __('label.success_add_question')]);
            } else {
                return response()->json(['status' => 400, 'errors' => __('label.error_add_question')]);
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function question_update(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'question' => 'required',
                'option_a' => 'required',
                'option_b' => 'required',
                'option_c' => 'required',
                'option_d' => 'required',
                'correct_answer' => 'required',
                'image' => 'image|mimes:jpeg,png,jpg|max:5120',
            ]);
            if ($validator->fails()) {
                $errs = $validator->errors()->all();
                return response()->json(['status' => 400, 'errors' => $errs]);
            }

            $question = Contest_Question::where('id', $request['id'])->first();
            if (isset($question)) {

                if (isset($request['image'])) {
                    $file = $request['image'];
                    $question['image'] = $this->common->saveImage($file, $this->folder, 'contest_question_');

                    $this->common->deleteImageToFolder($this->folder, basename($request['old_image']));
                }
                $question['question'] = $request['question'];
                $question['option_a'] = $request['option_a'];
                $question['option_b'] = $request['option_b'];
                $question['option_c'] = $request['option_c'];
                $question['option_d'] = $request['option_d'];
                $question['correct_answer'] = $request['correct_answer'];
                $question['note'] = $request['note'] ?? '';
                if ($question->save()) {
                    return response()->json(['status' => 200, 'success' => __('label.success_edit_question')]);
                } else {
                    return response()->json(['status' => 400, 'errors' => __('label.error_edit_question')]);
                }
            } else {
                return response()->json(['status' => 400, 'errors' => __('label.error_edit_question')]);
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function question_delete($id)
    {
        try {

            $question = Contest_Question::where('id', $id)->first();
            if (isset($question)) {
                $this->common->deleteImageToFolder($this->folder, $question['image']);
                $question->delete();
            }
            return redirect()->route('admin.contests.question.index', $id)->with('success', __('label.question_delete'));
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function question_show($id)
    {
        try {

            $data = Contest_Question::where('id', $id)->first();
            if (isset($data)) {

                $data['status'] = $data['status'] === 1 ? 0 : 1;
                $data->save();
                return response()->json(['status' => 200, 'success' => __('label.status_changed'), 'status_code' => $data->status]);
            } else {
                return response()->json(['status' => 400, 'errors' => __('label.data_not_found')]);
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    // Import-Export
    public function export(Request $request)
    {
        try {

            return Excel::download(new Contest_Question_Export(), 'Contest-Question-Formate.csv');
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function import(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'contest_id' => 'required|numeric',
                'import_file' => 'required|file|mimetypes:text/plain,text/csv,application/csv',
            ]);
            if ($validator->fails()) {
                $errs = $validator->errors()->all();
                return response()->json(['status' => 400, 'errors' => $errs]);
            }

            Excel::import(new Contest_Question_Import($request['contest_id']), $request->file('import_file'));

            return response()->json(['status' => 200, 'success' => __('label.success_import_file')]);
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    // Make Winner
    public function make_winner($contestId)
    {
        try {

            $contestData = Contest::where('id', $contestId)->first();
            if ($contestData) {

                $prizeJson = json_decode($contestData['prize_json'], true);

                // Step 1: Get users sorted by percentage desc
                $users = Contest_Leaderboard::where('contest_id', $contestId)->orderByDesc('percentage')->orderBy('created_at', 'asc')->get();

                // Step 2: Loop and assign ranks
                $rank = 1;

                foreach ($users as $user) {
                    // Find prize for this rank
                    $coin = 0;

                    foreach ($prizeJson as $group) {
                        if (strpos($group['rank'], '-') !== false) {
                            [$start, $end] = explode('-', $group['rank']);
                            if ($rank >= (int)$start && $rank <= (int)$end) {
                                $coin = $group['winning_coin'];
                                break;
                            }
                        } else {
                            if ((int)$group['rank'] === $rank) {
                                $coin = $group['winning_coin'];
                                break;
                            }
                        }
                    }

                    // Update user result
                    $user['rank'] = $rank;
                    $user['winning_coin'] = $coin;
                    $user->save();

                    User::where('id', $user['user_id'])->increment('wallet_coin', $coin);

                    $rank++;
                }

                return redirect()->route('admin.contests.index')->with('success', __('label.winners_declared'));
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    // Leaderboard
    public function leaderboard($contestId, Request $request)
    {
        try {
            $params['contest_id'] = $contestId;

            if ($request->ajax()) {

                $data = Contest_Leaderboard::where('contest_id', $contestId)->orderby('rank', 'asc')->with('user')->get();
                foreach ($data as $key => $value) {

                    if ($value['user'] != null) {
                        $this->common->userImage(array($value['user']));
                    } else {
                        $value['user'] = [
                            'image' => asset('assets/imgs/default.png')
                        ];
                    }
                }
                return DataTables()::of($data)->addIndexColumn()->make(true);
            }
            return view('admin.contest.leaderboard', $params);
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
}
